// Copyright 2005 Ben Hutchings <ben@decadentplace.org.uk>.
// See the file "COPYING" for licence details.

#include <cassert>
#include <exception>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <memory>
#include <queue>
#include <set>
#include <string>

#include <stdlib.h>
#include <unistd.h>

#include <gdkmm/pixbuf.h>
#include <gtkmm/main.h>
#include <gtkmm/window.h>

#include <imglib2/ImageErrors.h>
#include <nsGUIEvent.h>
#include <nsIBoxObject.h>
#include <nsIContent.h>
#include <nsIDocShell.h>
#include <nsIDOMAbstractView.h>
#include <nsIDOMDocumentEvent.h>
#include <nsIDOMDocumentView.h>
#include <nsIDOMElement.h>
#include <nsIDOMEventTarget.h>
#include <nsIDOMHTMLDocument.h>
#include <nsIDOMMouseEvent.h>
#include <nsIDOMNSDocument.h>
#include <nsIDOMWindow.h>
#include <nsIEventStateManager.h>
#include <nsIInterfaceRequestorUtils.h>
#include <nsIURI.h> // required before nsILink.h
#include <nsILink.h>
#include <nsIPresContext.h>
#include <nsIPresShell.h>
#include <nsIWebBrowser.h>
#include <nsString.h>

#include "browserwidget.hpp"
#include "childiterator.hpp"
#include "dvd.hpp"
#include "framebuffer.hpp"
#include "linkiterator.hpp"
#include "pixbufs.hpp"
#include "stylesheets.hpp"
#include "video.hpp"
#include "xpcom_support.hpp"

using xpcom_support::check;

namespace
{
    struct rectangle
    {
	int left, top;     // inclusive
	int right, bottom; // exclusive

	rectangle operator|=(const rectangle & other)
	    {
		if (other.empty())
		{
		    // use current extents unchanged
		}
		else if (empty())
		{
		    // use other extents
		    *this = other;
		}
		else
		{
		    // find rectangle enclosing both extents
		    left = std::min(left, other.left);
		    top = std::min(top, other.top);
		    right = std::max(right, other.right);
		    bottom = std::max(bottom, other.bottom);
		}

		return *this;
	    }

	rectangle operator&=(const rectangle & other)
	    {
		// find rectangle enclosed in both extents
		left = std::max(left, other.left);
		top = std::max(top, other.top);
		right = std::max(left, std::min(right, other.right));
		bottom = std::max(top, std::min(bottom, other.bottom));
		return *this;
	    }

	bool empty() const
	    {
		return left == right || bottom == top;
	    }
    };

    rectangle get_elem_rect(nsIDOMNSDocument * ns_doc,
			    nsIDOMElement * elem)
    {
	rectangle result;

	nsCOMPtr<nsIBoxObject> box;
	check(ns_doc->GetBoxObjectFor(elem, getter_AddRefs(box)));
	int width, height;
	check(box->GetScreenX(&result.left));
	check(box->GetScreenY(&result.top));
	check(box->GetWidth(&width));
	check(box->GetHeight(&height));
	result.right = result.left + width;
	result.bottom = result.top + height;

	for (ChildIterator it = ChildIterator(elem), end; it != end; ++it)
	{
	    nsCOMPtr<nsIDOMNode> child_node(*it);
	    PRUint16 child_type;
	    if (check(child_node->GetNodeType(&child_type)),
		child_type == nsIDOMNode::ELEMENT_NODE)
	    {
		nsCOMPtr<nsIDOMElement> child_elem(
		    do_QueryInterface(child_node));
		result |= get_elem_rect(ns_doc, child_elem);
	    }
	}

	return result;
    }

    class WebDvdWindow : public Gtk::Window
    {
    public:
	WebDvdWindow(int width, int height);
	void add_page(const std::string & uri);

    private:
	void add_video(const std::string & uri);
	void load_next_page();
	void on_net_state_change(const char * uri, gint flags, guint status);
	void save_screenshot();
	void process_links(nsIPresShell * pres_shell,
			   nsIPresContext * pres_context,
			   nsIDOMWindow * dom_window);
	void generate_dvdauthor_file();

	enum ResourceType { page_resource, video_resource };
	typedef std::pair<ResourceType, int> ResourceEntry;
	int width_, height_;
	BrowserWidget browser_widget_;
	nsCOMPtr<nsIStyleSheet> stylesheet_;
	std::queue<std::string> page_queue_;
	std::map<std::string, ResourceEntry> resource_map_;
	std::vector<std::vector<std::string> > page_links_;
	std::vector<std::string> video_paths_;
	bool loading_;
	int pending_req_count_;
	struct link_state;
	std::auto_ptr<link_state> link_state_;
    };

    WebDvdWindow::WebDvdWindow(int width, int height)
	    : width_(width), height_(height),
	      stylesheet_(load_css("file://" WEBDVD_LIB_DIR "/webdvd.css")),
	      loading_(false),
	      pending_req_count_(0)
    {
	set_default_size(width, height);
	add(browser_widget_);
	browser_widget_.show();
	browser_widget_.signal_net_state().connect(
	    SigC::slot(*this, &WebDvdWindow::on_net_state_change));
    }

    void WebDvdWindow::add_page(const std::string & uri)
    {
	if (resource_map_.insert(
		std::make_pair(uri, ResourceEntry(page_resource, 0)))
	    .second)
	{
	    page_queue_.push(uri);
	    if (!loading_)
		load_next_page();
	}
    }

    void WebDvdWindow::add_video(const std::string & uri)
    {
	if (resource_map_.insert(
		std::make_pair(uri, ResourceEntry(video_resource,
						  video_paths_.size() + 1)))
	    .second)
	{
	    // FIXME: Should accept some slightly different URI prefixes
	    // (e.g. file://localhost/) and decode any URI-escaped
	    // characters in the path.
	    assert(uri.compare(0, 8, "file:///") == 0);
	    video_paths_.push_back(uri.substr(7));
	}
    }

    void WebDvdWindow::load_next_page()
    {
	loading_ = true;

	assert(!page_queue_.empty());
	const std::string & uri = page_queue_.front();
	std::cout << "loading " << uri << std::endl;

	std::size_t page_count = page_links_.size();
	resource_map_[uri].second = ++page_count;
	page_links_.resize(page_count);
	browser_widget_.load_uri(uri);
    }

    void WebDvdWindow::on_net_state_change(const char * uri,
					   gint flags, guint status)
    {
	enum {
	    process_nothing,
	    process_new_page,
	    process_current_link
	} action = process_nothing;

	if (flags & GTK_MOZ_EMBED_FLAG_IS_REQUEST)
	{
	    if (flags & GTK_MOZ_EMBED_FLAG_START)
		++pending_req_count_;
	    if (flags & GTK_MOZ_EMBED_FLAG_STOP)
	    {
		assert(pending_req_count_ != 0);
		--pending_req_count_;
	    }
	    if (pending_req_count_ == 0 && link_state_.get())
		action = process_current_link;
	}
	    
	if (flags & GTK_MOZ_EMBED_FLAG_STOP
	    && flags & GTK_MOZ_EMBED_FLAG_IS_WINDOW)
	    action = process_new_page;

	if (action != process_nothing)
	{
	    assert(loading_ && !page_queue_.empty());
	    assert(pending_req_count_ == 0);

	    try
	    {
		// Check whether the load was successful, ignoring this
		// pseudo-error.
		if (status != NS_IMAGELIB_ERROR_LOAD_ABORTED)
		    check(status);

		nsCOMPtr<nsIWebBrowser> browser(
		    browser_widget_.get_browser());
		nsCOMPtr<nsIDocShell> doc_shell(do_GetInterface(browser));
		assert(doc_shell);
		nsCOMPtr<nsIPresShell> pres_shell;
		check(doc_shell->GetPresShell(getter_AddRefs(pres_shell)));
		nsCOMPtr<nsIPresContext> pres_context;
		check(doc_shell->GetPresContext(
			  getter_AddRefs(pres_context)));
		nsCOMPtr<nsIDOMWindow> dom_window;
		check(browser->GetContentDOMWindow(
			  getter_AddRefs(dom_window)));

		if (action == process_new_page)
		{
		    apply_style_sheet(stylesheet_, pres_shell);
		    save_screenshot();
		}
		process_links(pres_shell, pres_context, dom_window);
		if (!link_state_.get())
		{
		    page_queue_.pop();
		    if (page_queue_.empty())
		    {
			generate_dvdauthor_file();
			Gtk::Main::quit();
		    }
		    else
			load_next_page();
		}
	    }
	    catch (std::exception & e)
	    {
		std::cerr << "Fatal error";
		if (!page_queue_.empty())
		    std::cerr << " while processing <" << page_queue_.front()
			      << ">";
		std::cerr << ": " << e.what() << "\n";
		Gtk::Main::quit();
	    }
	}
    }

    void WebDvdWindow::save_screenshot()
    {
	char filename[25];
	std::sprintf(filename, "page_%06d_back.png", page_links_.size());
	Glib::RefPtr<Gdk::Window> window(get_window());
	assert(window);
	window->process_updates(true);
	std::cout << "saving " << filename << std::endl;
	Gdk::Pixbuf::create(Glib::RefPtr<Gdk::Drawable>(window),
			    window->get_colormap(),
			    0, 0, 0, 0, width_, height_)
	    ->save(filename, "png");
    }

    struct WebDvdWindow::link_state
    {
	Glib::RefPtr<Gdk::Pixbuf> diff_pixbuf;

	std::ofstream spumux_file;

	int link_num;
	LinkIterator links_it, links_end;

	rectangle link_rect;
	bool link_changing;
	Glib::RefPtr<Gdk::Pixbuf> norm_pixbuf;
    };

    void WebDvdWindow::process_links(nsIPresShell * pres_shell,
				     nsIPresContext * pres_context,
				     nsIDOMWindow * dom_window)
    {
	Glib::RefPtr<Gdk::Window> window(get_window());
	assert(window);

	nsCOMPtr<nsIDOMDocument> basic_doc;
	check(dom_window->GetDocument(getter_AddRefs(basic_doc)));
	nsCOMPtr<nsIDOMNSDocument> ns_doc(do_QueryInterface(basic_doc));
	assert(ns_doc);
	nsCOMPtr<nsIEventStateManager> event_state_man(
	    pres_context->EventStateManager()); // does not AddRef
	assert(event_state_man);
	nsCOMPtr<nsIDOMDocumentEvent> event_factory(
	    do_QueryInterface(basic_doc));
	assert(event_factory);
	nsCOMPtr<nsIDOMDocumentView> doc_view(do_QueryInterface(basic_doc));
	assert(doc_view);
	nsCOMPtr<nsIDOMAbstractView> view;
	check(doc_view->GetDefaultView(getter_AddRefs(view)));

	// Set up or recover our iteration state.
	std::auto_ptr<link_state> state(link_state_);
	if (!state.get())
	{
	    state.reset(new link_state);

	    state->diff_pixbuf = Gdk::Pixbuf::create(Gdk::COLORSPACE_RGB,
						     true, // has_alpha
						     8, // bits_per_sample
						     width_, height_);

	    char spumux_filename[20];
	    std::sprintf(spumux_filename,
			 "page_%06d.spumux", page_links_.size());
	    state->spumux_file.open(spumux_filename);
	    state->spumux_file <<
		"<subpictures>\n"
		"  <stream>\n"
		"    <spu force='yes' start='00:00:00.00'\n"
		"        highlight='page_" << std::setfill('0')
			       << std::setw(6) << page_links_.size()
			       << "_links.png'\n"
		"        select='page_" << std::setfill('0')
			       << std::setw(6) << page_links_.size()
			       << "_links.png'>\n";

	    state->link_num = 0;
	    state->links_it = LinkIterator(basic_doc);
	    state->link_changing = false;
	}
	    
	rectangle window_rect = { 0, 0, width_, height_ };

	for (/* no initialisation */;
	     state->links_it != state->links_end;
	     ++state->links_it)
	{
	    nsCOMPtr<nsIDOMNode> node(*state->links_it);

	    // Find the link URI.
	    nsCOMPtr<nsILink> link(do_QueryInterface(node));
	    assert(link);
	    nsCOMPtr<nsIURI> uri;
	    check(link->GetHrefURI(getter_AddRefs(uri)));
	    std::string uri_string;
	    {
		nsCString uri_ns_string;
		check(uri->GetSpec(uri_ns_string));
		uri_string.assign(uri_ns_string.BeginReading(),
				  uri_ns_string.EndReading());
	    }
	    std::string uri_sans_fragment(uri_string, 0, uri_string.find('#'));

	    // Is this a new link?
	    if (!state->link_changing)
	    {
		// Find a rectangle enclosing the link and clip it to the
		// window.
		nsCOMPtr<nsIDOMElement> elem(do_QueryInterface(node));
		assert(elem);
		state->link_rect = get_elem_rect(ns_doc, elem);
		state->link_rect &= window_rect;

		if (state->link_rect.empty())
		{
		    std::cerr << "Ignoring invisible link to "
			      << uri_string << "\n";
		    continue;
		}

		++state->link_num;

		if (state->link_num >= dvd::menu_buttons_max)
		{
		    if (state->link_num == dvd::menu_buttons_max)
			std::cerr << "No more than " << dvd::menu_buttons_max
				  << " buttons can be placed on a page\n";
		    std::cerr << "Ignoring link to " << uri_string << "\n";
		    continue;
		}

		// Check whether this is a link to a video or a page then
		// add it to the known resources if not already seen.
	        nsCString path;
		check(uri->GetPath(path));
		// FIXME: This is a bit of a hack.  Perhaps we could decide
		// later based on the MIME type determined by Mozilla?
		if (path.Length() > 4
		    && std::strcmp(path.EndReading() - 4, ".vob") == 0)
		{
		    PRBool is_file;
		    check(uri->SchemeIs("file", &is_file));
		    if (!is_file)
		    {
			std::cerr << "Links to video must use the file:"
				  << " scheme\n";
			continue;
		    }
		    add_video(uri_sans_fragment);
		}
		else
		{
		    add_page(uri_sans_fragment);
		}

		nsCOMPtr<nsIContent> content(do_QueryInterface(node));
		assert(content);
		nsCOMPtr<nsIDOMEventTarget> event_target(
		    do_QueryInterface(node));
		assert(event_target);

		state->norm_pixbuf = Gdk::Pixbuf::create(
		    Glib::RefPtr<Gdk::Drawable>(window),
		    window->get_colormap(),
		    state->link_rect.left,
		    state->link_rect.top,
		    0,
		    0,
		    state->link_rect.right - state->link_rect.left,
		    state->link_rect.bottom - state->link_rect.top);

		nsCOMPtr<nsIDOMEvent> event;
		check(event_factory->CreateEvent(
			  NS_ConvertASCIItoUTF16("MouseEvents"),
			  getter_AddRefs(event)));
		nsCOMPtr<nsIDOMMouseEvent> mouse_event(
		    do_QueryInterface(event));
		assert(mouse_event);
		check(mouse_event->InitMouseEvent(
			  NS_ConvertASCIItoUTF16("mouseover"),
			  true,  // can bubble
			  true,  // cancelable
			  view,
			  0,     // detail: mouse click count
			  state->link_rect.left, // screenX
			  state->link_rect.top,  // screenY
			  state->link_rect.left, // clientX
			  state->link_rect.top,  // clientY
			  false, false, false, false, // qualifiers
			  0,     // button: left (or primary)
			  0));   // related target
		PRBool dummy;
		check(event_target->DispatchEvent(mouse_event,
						  &dummy));
		check(event_state_man->SetContentState(content,
						       NS_EVENT_STATE_HOVER));

		pres_shell->FlushPendingNotifications(true);

		// We may have to exit and wait for image loading
		// to complete, at which point we will be called
		// again.
		if (pending_req_count_ > 0)
		{
		    state->link_changing = true;
		    link_state_ = state;
		    return;
		}
	    }

	    window->process_updates(true);

	    Glib::RefPtr<Gdk::Pixbuf> changed_pixbuf(
		Gdk::Pixbuf::create(
		    Glib::RefPtr<Gdk::Drawable>(window),
		    window->get_colormap(),
		    state->link_rect.left,
		    state->link_rect.top,
		    0,
		    0,
		    state->link_rect.right - state->link_rect.left,
		    state->link_rect.bottom - state->link_rect.top));
	    diff_rgb_pixbufs(
		state->norm_pixbuf,
		changed_pixbuf,
		state->diff_pixbuf,
		state->link_rect.left,
		state->link_rect.top,
		state->link_rect.right - state->link_rect.left,
		state->link_rect.bottom - state->link_rect.top);

	    state->spumux_file <<
		"      <button x0='" << state->link_rect.left << "'"
		" y0='" << state->link_rect.top << "'"
		" x1='" << state->link_rect.right - 1 << "'"
		" y1='" << state->link_rect.bottom - 1 << "'/>\n";

	    // Add to the page's links, ignoring any fragment (for now).
	    page_links_.back().push_back(uri_sans_fragment);
	}

	quantise_rgba_pixbuf(state->diff_pixbuf, dvd::button_n_colours);

	char filename[25];
	std::sprintf(filename, "page_%06d_links.png", page_links_.size());
	std::cout << "saving " << filename << std::endl;
	state->diff_pixbuf->save(filename, "png");

	state->spumux_file <<
	    "    </spu>\n"
	    "  </stream>\n"
	    "</subpictures>\n";
    }

    void generate_page_dispatch(std::ostream &, int indent,
				int first_page, int last_page);

    void WebDvdWindow::generate_dvdauthor_file()
    {
	std::ofstream file("webdvd.dvdauthor");

	// We generate code that uses registers in the following way:
	//
	// g0:     link destination (when jumping to menu 1), then scratch
	// g1:     current location
	// g2-g11: location history (g2 = most recent)
	// g12:    location that last linked to a video
	//
	// All locations are divided into two bitfields: the least
	// significant 10 bits are a page/menu number and the most
	// significant 6 bits are a link/button number.  This is
	// chosen for compatibility with the encoding of the s8
	// (button) register.
	//
	static const int link_mult = dvd::reg_s8_button_mult;
	static const int page_mask = link_mult - 1;
	static const int link_mask = (1 << dvd::reg_bits) - link_mult;

	file <<
	    "<dvdauthor>\n"
	    "  <vmgm>\n"
	    "    <menus>\n";
	    
	for (std::size_t page_num = 1;
	     page_num <= page_links_.size();
	     ++page_num)
	{
	    std::vector<std::string> & page_links =
		page_links_[page_num - 1];

	    if (page_num == 1)
	    {
		// This is the first page (root menu) which needs to
		// include initialisation and dispatch code.
	
		file <<
		    "      <pgc entry='title'>\n"
		    "        <pre>\n"
		    // Has the location been set yet?
		    "          if (g1 eq 0)\n"
		    "          {\n"
		    // Initialise the current location to first link on
		    // this page.
		    "            g1 = " << 1 * link_mult + 1 << ";\n"
		    "          }\n"
		    "          else\n"
		    "          {\n"
		    // Has the user selected a link?
		    "            if (g0 ne 0)\n"
		    "            {\n"
		    // First update the history.
                    // Does link go to the last page in the history?
		    "              if (((g0 ^ g2) &amp; " << page_mask
		     << ") == 0)\n"
                    // It does; we treat this as going back and pop the old
		    // location off the history stack into the current
		    // location.  Clear the free stack slot.
		    "              {\n"
		    "                g1 = g2; g2 = g3; g3 = g4; g4 = g5;\n"
		    "                g5 = g6; g6 = g7; g7 = g8; g8 = g9;\n"
		    "                g9 = g10; g10 = g11; g11 = 0;\n"
	            "              }\n"
                    "              else\n"
		    // Link goes to some other page, so push current
		    // location onto the history stack and set the current
		    // location to be exactly the target location.
		    "              {\n"
		    "                g11 = g10; g10 = g9; g9 = g8; g8 = g7;\n"
		    "                g7 = g6; g6 = g5; g5 = g4; g4 = g3;\n"
		    "                g3 = g2; g2 = g1; g1 = g0;\n"
	            "              }\n"
		    "            }\n"
		    // Find the target page number.
		    "            g0 = g1 &amp; " << page_mask << ";\n";
		// There seems to be no way to perform a computed jump,
		// so we generate all possible jumps and a binary search
		// to select the correct one.
		generate_page_dispatch(file, 12, 1, page_links_.size());
		file <<
		    "          }\n";
	    }
	    else // page_num != 1
	    {
		file <<
		    "      <pgc>\n"
		    "        <pre>\n";
	    }

	    file <<
		// Clear link indicator and highlight the
		// appropriate link/button.
		"          g0 = 0; s8 = g1 &amp; " << link_mask << ";\n"
		"        </pre>\n"
	        "        <vob file='page_"
		 << std::setfill('0') << std::setw(6) << page_num
		 << ".vob'/>\n";

	    for (std::size_t link_num = 1;
		 link_num <= page_links.size();
		 ++link_num)
	    {
		file <<
		    "        <button>"
		    // Update current location.
		    " g1 = " << link_num * link_mult + page_num << ";";

		// Jump to appropriate resource.
		const ResourceEntry & resource_loc =
		    resource_map_[page_links[link_num - 1]];
		if (resource_loc.first == page_resource)
		    file <<
			" g0 = " << 1 * link_mult + resource_loc.second << ";"
			" jump menu 1;";
		else if (resource_loc.first == video_resource)
		    file << " jump title " << resource_loc.second << ";";

		file <<  " </button>\n";
	    }

	    file << "      </pgc>\n";
	}

	file <<
	    "    </menus>\n"
	    "  </vmgm>\n";

	// Generate a titleset for each video.  This appears to make
	// jumping to titles a whole lot simpler.
	for (std::size_t video_num = 1;
	     video_num <= video_paths_.size();
	     ++video_num)
	{
	    file <<
		"  <titleset>\n"
		// Generate a dummy menu so that the menu button on the
		// remote control will work.
		"    <menus>\n"
		"      <pgc entry='root'>\n"
		"        <pre> jump vmgm menu; </pre>\n"
		"      </pgc>\n"
		"    </menus>\n"
		"    <titles>\n"
		"      <pgc>\n"
		// Record calling page/menu.
		"        <pre> g12 = g1; </pre>\n"
		// FIXME: Should XML-escape the path
		"        <vob file='" << video_paths_[video_num - 1]
		 << "'/>\n"
		// If page/menu location has not been changed during the
		// video, change the location to be the following
		// link/button when returning to it.  In any case,
		// return to a page/menu.
		"        <post> if (g1 eq g12) g1 = g1 + " << link_mult
		 << "; call menu; </post>\n"
		"      </pgc>\n"
		"    </titles>\n"
		"  </titleset>\n";
	}

	file <<
	    "</dvdauthor>\n";
    }

    void generate_page_dispatch(std::ostream & file, int indent,
				int first_page, int last_page)
    {
	if (first_page == 1 && last_page == 1)
	{
	    // The dispatch code is *on* page 1 so we must not dispatch to
	    // page 1 since that would cause an infinite loop.  This case
	    // should be unreachable if there is more than one page due
	    // to the following case.
	}
	else if (first_page == 1 && last_page == 2)
	{
	    // dvdauthor doesn't allow empty blocks or null statements so
	    // when selecting between pages 1 and 2 we don't use an "else"
	    // part.  We must use braces so that a following "else" will
	    // match the right "if".
	    file << std::setw(indent) << "" << "{\n"
		 << std::setw(indent) << "" << "if (g0 eq 2)\n"
		 << std::setw(indent + 2) << "" << "jump menu 2;\n"
		 << std::setw(indent) << "" << "}\n";
	}
	else if (first_page == last_page)
	{
	    file << std::setw(indent) << ""
		 << "jump menu " << first_page << ";\n";
	}
	else
	{
	    int middle = (first_page + last_page) / 2;
	    file << std::setw(indent) << "" << "if (g0 le " << middle << ")\n";
	    generate_page_dispatch(file, indent + 2, first_page, middle);
	    file << std::setw(indent) << "" << "else\n";
	    generate_page_dispatch(file, indent + 2, middle + 1, last_page);
	}
    }

} // namespace

int main(int argc, char ** argv)
{
    // Get dimensions
    int width = video::pal_oscan_width, height = video::pal_oscan_height;
    for (int i = 1; i < argc - 1; ++i)
	if (std::strcmp(argv[i], "-geometry") == 0)
	{
	    std::sscanf(argv[i + 1], "%dx%d", &width, &height);
	    break;
	}
    // A depth of 24 results in 8 bits each for RGB components, which
    // translates into "enough" bits for YUV components.
    const int depth = 24;

    try
    {
	// Spawn Xvfb and set env variables so that Xlib will use it
	FrameBuffer fb(width, height, depth);
	setenv("XAUTHORITY", fb.get_x_authority().c_str(), true);
	setenv("DISPLAY", fb.get_x_display().c_str(), true);

	// Initialise Gtk and Mozilla
	Gtk::Main kit(argc, argv);
	BrowserWidget::init();

	WebDvdWindow window(width, height);
	for (int argi = 1; argi < argc; ++argi)
	    window.add_page(argv[argi]);
	if (argc < 2)
	    window.add_page("about:");
	Gtk::Main::run(window);
    }
    catch (std::exception & e)
    {
	std::cerr << "Fatal error: " << e.what() << "\n";
	return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}
